------------------------------------------------------------------------
-- Event:        Delphi Day 2018, Piacenza, June 06 2018               -
--               https://www.delphiday.it/                             -
-- Seminary:     How to write high performance queries in T-SQL        -
-- Demo:         Setup                                                 -
-- Author:       Sergio Govoni                                         -
-- Notes:        --                                                    -
------------------------------------------------------------------------


-- Full backup of WideWorldImporters sample database is available on GitHub
-- https://github.com/Microsoft/sql-server-samples/releases/tag/wide-world-importers-v1.0


-- Documentation about WideWorldImporters sample database for SQL Server
-- and Azure SQL Database
-- https://github.com/Microsoft/sql-server-samples/tree/master/samples/databases/wide-world-importers


USE [master];

-- Drop Database
IF (DB_ID('WideWorldImporters') IS NOT NULL)
BEGIN
  ALTER DATABASE [WideWorldImporters]
    SET SINGLE_USER WITH ROLLBACK IMMEDIATE;

  DROP DATABASE [WideWorldImporters];
END;
GO

-- Restore DB
RESTORE DATABASE [WideWorldImporters]
  FROM  DISK = N'C:\SQL\DBs\Backup\WideWorldImporters-Full.bak'
  WITH
    FILE = 1
    ,MOVE N'WWI_Primary' TO N'C:\SQL\DBs\WideWorldImporters.mdf'
    ,MOVE N'WWI_UserData' TO N'C:\SQL\DBs\WideWorldImporters_UserData.ndf'
    ,MOVE N'WWI_Log' TO N'C:\SQL\DBs\WideWorldImporters.ldf'
    ,MOVE N'WWI_InMemory_Data_1' TO N'C:\SQL\DBs\WideWorldImporters_InMemory_Data_1'
    ,NOUNLOAD
    ,STATS = 5;
GO


USE [WideWorldImporters]
GO


-- Create table Application.Employees
CREATE TABLE Application.Employees
(
  empid INTEGER NOT NULL IDENTITY,
  lastname NVARCHAR(20) NOT NULL,
  firstname NVARCHAR(10) NOT NULL,
  title NVARCHAR(30) NOT NULL,
  titleofcourtesy NVARCHAR(25) NOT NULL,
  birthdate DATETIME NOT NULL,
  hiredate DATETIME NOT NULL,
  address NVARCHAR(60) NOT NULL,
  city NVARCHAR(15) NOT NULL,
  region NVARCHAR(15) NULL,
  postalcode NVARCHAR(10) NULL,
  country NVARCHAR(15) NOT NULL,
  phone NVARCHAR(24) NOT NULL,
  mgrid INTEGER NULL,

  CONSTRAINT PK_Employees PRIMARY KEY(empid),
  CONSTRAINT FK_Employees_Employees FOREIGN KEY(mgrid)
    REFERENCES Application.Employees(empid),
  CONSTRAINT CHK_birthdate CHECK(birthdate <= CURRENT_TIMESTAMP)
);

CREATE NONCLUSTERED INDEX idx_nc_lastname ON Application.Employees(lastname);
CREATE NONCLUSTERED INDEX idx_nc_postalcode ON Application.Employees(postalcode);

SET NOCOUNT ON;

-- Populate table Application.Employees
SET IDENTITY_INSERT Application.Employees ON;

INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(1, N'Davis', N'Sara', N'CEO', N'Ms.', '19581208 00:00:00.000', '20020501 00:00:00.000', N'7890 - 20th Ave. E., Apt. 2A', N'Seattle', N'WA', N'10003', N'USA', N'(206) 555-0101', NULL);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(2, N'Funk', N'Don', N'Vice President, Sales', N'Dr.', '19620219 00:00:00.000', '20020814 00:00:00.000', N'9012 W. Capital Way', N'Tacoma', N'WA', N'10001', N'USA', N'(206) 555-0100', 1);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(3, N'Lew', N'Judy', N'Sales Manager', N'Ms.', '19730830 00:00:00.000', '20020401 00:00:00.000', N'2345 Moss Bay Blvd.', N'Kirkland', N'WA', N'10007', N'USA', N'(206) 555-0103', 2);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(4, N'Peled', N'Yael', N'Sales Representative', N'Mrs.', '19470919 00:00:00.000', '20030503 00:00:00.000', N'5678 Old Redmond Rd.', N'Redmond', N'WA', N'10009', N'USA', N'(206) 555-0104', 3);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(5, N'Buck', N'Sven', N'Sales Manager', N'Mr.', '19650304 00:00:00.000', '20031017 00:00:00.000', N'8901 Garrett Hill', N'London', NULL, N'10004', N'UK', N'(71) 234-5678', 2);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(6, N'Suurs', N'Paul', N'Sales Representative', N'Mr.', '19730702 00:00:00.000', '20031017 00:00:00.000', N'3456 Coventry House, Miner Rd.', N'London', NULL, N'10005', N'UK', N'(71) 345-6789', 5);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(7, N'King', N'Russell', N'Sales Representative', N'Mr.', '19700529 00:00:00.000', '20040102 00:00:00.000', N'6789 Edgeham Hollow, Winchester Way', N'London', NULL, N'10002', N'UK', N'(71) 123-4567', 5);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(8, N'Cameron', N'Maria', N'Sales Representative', N'Ms.', '19680109 00:00:00.000', '20040305 00:00:00.000', N'4567 - 11th Ave. N.E.', N'Seattle', N'WA', N'10006', N'USA', N'(206) 555-0102', 3);
INSERT INTO Application.Employees(empid, lastname, firstname, title, titleofcourtesy, birthdate, hiredate, address, city, region, postalcode, country, phone, mgrid)
  VALUES(9, N'Dolgopyatova', N'Zoya', N'Sales Representative', N'Ms.', '19760127 00:00:00.000', '20041115 00:00:00.000', N'1234 Houndstooth Rd.', N'London', NULL, N'10008', N'UK', N'(71) 456-7890', 5);

SET IDENTITY_INSERT Application.Employees OFF;
GO


CREATE TABLE [Warehouse].[StockItemTransactionsNoCCI](
	[StockItemTransactionID] [int] NOT NULL,
	[StockItemID] [int] NOT NULL,
	[TransactionTypeID] [int] NOT NULL,
	[CustomerID] [int] NULL,
	[InvoiceID] [int] NULL,
	[SupplierID] [int] NULL,
	[PurchaseOrderID] [int] NULL,
	[TransactionOccurredWhen] [datetime2](7) NOT NULL,
	[Quantity] [decimal](18, 3) NOT NULL,
	[LastEditedBy] [int] NOT NULL,
	[LastEditedWhen] [datetime2](7) NOT NULL,
 CONSTRAINT [PK_Warehouse_StockItemTransactionsNoCCI] PRIMARY KEY NONCLUSTERED 
(
	[StockItemTransactionID] ASC
)WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [USERDATA]
) ON [USERDATA]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] ADD  CONSTRAINT [DF_Warehouse_StockItemTransactionsNoCCI_StockItemTransactionID]  DEFAULT (NEXT VALUE FOR [Sequences].[TransactionID]) FOR [StockItemTransactionID]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] ADD  CONSTRAINT [DF_Warehouse_StockItemTransactionsNoCCI_LastEditedWhen]  DEFAULT (sysdatetime()) FOR [LastEditedWhen]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_Application_People] FOREIGN KEY([LastEditedBy])
REFERENCES [Application].[People] ([PersonID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_Application_People]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_CustomerID_Sales_Customers] FOREIGN KEY([CustomerID])
REFERENCES [Sales].[Customers] ([CustomerID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_CustomerID_Sales_Customers]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_InvoiceID_Sales_Invoices] FOREIGN KEY([InvoiceID])
REFERENCES [Sales].[Invoices] ([InvoiceID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_InvoiceID_Sales_Invoices]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_PurchaseOrderID_Purchasing_PurchaseOrders] FOREIGN KEY([PurchaseOrderID])
REFERENCES [Purchasing].[PurchaseOrders] ([PurchaseOrderID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_PurchaseOrderID_Purchasing_PurchaseOrders]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_StockItemID_Warehouse_StockItems] FOREIGN KEY([StockItemID])
REFERENCES [Warehouse].[StockItems] ([StockItemID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_StockItemID_Warehouse_StockItems]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_SupplierID_Purchasing_Suppliers] FOREIGN KEY([SupplierID])
REFERENCES [Purchasing].[Suppliers] ([SupplierID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_SupplierID_Purchasing_Suppliers]
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI]  WITH CHECK ADD  CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_TransactionTypeID_Application_TransactionTypes] FOREIGN KEY([TransactionTypeID])
REFERENCES [Application].[TransactionTypes] ([TransactionTypeID])
GO

ALTER TABLE [Warehouse].[StockItemTransactionsNoCCI] CHECK CONSTRAINT [FK_Warehouse_StockItemTransactionsNoCCI_TransactionTypeID_Application_TransactionTypes]
GO

INSERT INTO [Warehouse].[StockItemTransactionsNoCCI]
(
  [StockItemTransactionID]
  ,[StockItemID]
  ,[TransactionTypeID]
  ,[CustomerID]
  ,[InvoiceID]
  ,[SupplierID]
  ,[PurchaseOrderID]
  ,[TransactionOccurredWhen]
  ,[Quantity]
  ,[LastEditedBy]
  ,[LastEditedWhen]
)
SELECT
  [StockItemTransactionID]
  ,[StockItemID]
  ,[TransactionTypeID]
  ,[CustomerID]
  ,[InvoiceID]
  ,[SupplierID]
  ,[PurchaseOrderID]
  ,[TransactionOccurredWhen]
  ,[Quantity]
  ,[LastEditedBy]
  ,[LastEditedWhen]
FROM
  [Warehouse].[StockItemTransactions]
GO

CREATE OR ALTER FUNCTION Website.DDay18CalculateCustomerPrice
(
    @BuyingGroupID INTEGER
    ,@StockItemID INTEGER
    ,@UnitPrice DECIMAL(18, 2)
    ,@PricingDate DATE
)
RETURNS DECIMAL(18,2)
WITH EXECUTE AS OWNER
AS
BEGIN
    DECLARE @CalculatedPrice DECIMAL(18,2);
 
    SET @CalculatedPrice = (
                             SELECT
                               --MIN(sd.DiscountPercentage)
                               ISNULL(@UnitPrice - ROUND(@UnitPrice * MAX(sd.DiscountPercentage) / 100., 2), @UnitPrice)
                             FROM
                               Sales.SpecialDeals AS sd
                             WHERE
                               ((sd.StockItemID = @StockItemID) OR (sd.StockItemID IS NULL))
                               AND ((sd.BuyingGroupID = @BuyingGroupID) OR (sd.BuyingGroupID IS NULL))
                               AND (@PricingDate BETWEEN sd.StartDate AND sd.EndDate)
                           );

    RETURN @CalculatedPrice;
END;
GO

CREATE OR ALTER VIEW Sales.SalesOrderHeader
AS
SELECT
  so.CustomerID
  ,OrderDate = MIN(so.OrderDate)
  ,TotalDue = SUM(sol.UnitPrice * sol.PickedQuantity)
FROM
  Sales.Orders AS so
JOIN
  Sales.OrderLines AS sol ON so.OrderID=sol.OrderID
GROUP BY
  so.CustomerID, so.OrderID;
GO

IF NOT EXISTS (SELECT *
               FROM sys.indexes
               WHERE object_id = OBJECT_ID('Warehouse.StockItems')
                 AND name = 'IX_Warehouse_StockItems_Size')
CREATE INDEX IX_Warehouse_StockItems_Size ON Warehouse.StockItems
(
  Size
);

IF NOT EXISTS (SELECT *
               FROM sys.indexes
               WHERE object_id = OBJECT_ID('Warehouse.StockItems')
                 AND name = 'IX_Warehouse_StockItems_ValidFrom')
CREATE INDEX IX_Warehouse_StockItems_ValidFrom ON Warehouse.StockItems
(
  ValidFrom
);
GO


-- AdventureWorks2017

------------------------------------------------------------------------
-- Download backup of AdventureWorks2017:                              -
-- https://www.microsoft.com/en-us/download/details.aspx?id=49502      -
------------------------------------------------------------------------

USE [master];
GO

-- Drop Database
IF (DB_ID('AdventureWorks2017') IS NOT NULL)
BEGIN
  ALTER DATABASE [AdventureWorks2017]
    SET SINGLE_USER WITH ROLLBACK IMMEDIATE;

  DROP DATABASE [AdventureWorks2017];
END;
GO

RESTORE DATABASE [AdventureWorks2017]
  FROM DISK = N'C:\SQL\DBs\Backup\AdventureWorks2017.bak'
  WITH
    FILE = 1
    ,MOVE N'AdventureWorks2017' TO N'C:\SQL\DBs\AdventureWorks2017.mdf'
    ,MOVE N'AdventureWorks2017_log' TO N'C:\SQL\DBs\AdventureWorks2017_log.ldf'
    ,NOUNLOAD
    ,STATS = 5;
GO

/*
ALTER AUTHORIZATION ON DATABASE::AdventureWorks2017 TO [MARCONI\AdventureWorks2017]
GO
*/

USE [AdventureWorks2017];
GO

------------------------------------------------------------------------
-- Stored procedures                                                   -
------------------------------------------------------------------------

-- dbo.usp_PO_Extra_Received
/*
IF OBJECT_ID('dbo.usp_PO_Extra_Received') IS NOT NULL
  DROP PROCEDURE dbo.usp_PO_Extra_Received;
GO
*/

CREATE OR ALTER PROCEDURE dbo.usp_PO_Extra_Received
AS
BEGIN
  SELECT
    *
  FROM
    Purchasing.PurchaseOrderDetail AS pod
  WHERE
    (pod.ReceivedQty >= pod.OrderQty)
  ORDER BY
    pod.RejectedQty DESC;
END;
GO

-- dbo.usp_StateProvinceByTerritory
/*
IF OBJECT_ID('dbo.usp_StateProvinceByTerritory') IS NOT NULL
  DROP PROCEDURE dbo.usp_StateProvinceByTerritory;
GO
*/

CREATE OR ALTER PROCEDURE dbo.usp_StateProvinceByTerritory
(
  @TerritoryName NVARCHAR(50)
)
AS
BEGIN
  SELECT
    p.StateProvinceID
	,p.Name
	,p.StateProvinceCode
	,p.CountryRegionCode
	,t.*
  FROM
    [Person].[StateProvince] AS p
  JOIN
    [Sales].[SalesTerritory] AS t
    ON p.TerritoryID = t.TerritoryID
  WHERE
    (t.Name = @TerritoryName)
  --OPTION (OPTIMIZE FOR (@TerritoryName UNKNOWN))
  --OPTION (OPTIMIZE FOR (@TerritoryName='Italy'))
END;
GO

------------------------------------------------------------------------
-- Tables                                                              -
------------------------------------------------------------------------

IF OBJECT_ID('myOrderHeader', 'U') IS NOT NULL
  DROP TABLE dbo.myOrderHeader;
GO

CREATE TABLE dbo.myOrderHeader
(
 OrderID INT IDENTITY(1, 1) NOT NULL
 ,OrderDate DATETIME DEFAULT GETDATE() NOT NULL
 ,CustomerID INT DEFAULT 1 NOT NULL
 ,ShipName VARCHAR(20) DEFAULT 'name'
 ,ShipAddress VARCHAR(40) DEFAULT 'address'
 ,ShipVia VARCHAR(40) DEFAULT 'via'
 ,ShipCity VARCHAR(20) DEFAULT 'city'
 ,ShipRegion VARCHAR(20) DEFAULT 'region'
 ,ShipPostalCode VARCHAR(20) DEFAULT 'postal code'
 ,ShipCountry VARCHAR(20) DEFAULT 'country'
 ,DeliveryDate DATETIME DEFAULT (GETDATE() + DATEPART(SS, GETDATE())) NOT NULL
 ,DeliveryNote VARCHAR(40)
);
GO

/*
SELECT * FROM dbo.myOrderHeader;
GO
*/

SET NOCOUNT ON;
GO

INSERT INTO dbo.myOrderHeader DEFAULT VALUES
GO 1000

SET NOCOUNT OFF;
GO

/*
DROP INDEX dbo.myOrderHeader.FIDX_myOrderHeader
*/

CREATE NONCLUSTERED INDEX FIDX_myOrderHeader on dbo.myOrderHeader
(
  [DeliveryDate]
)
WHERE
  (DeliveryDate >= '20150128');
GO


IF OBJECT_ID('dbo.OrderDetail', 'U') IS NOT NULL
  DROP TABLE dbo.OrderDetail;
GO

CREATE TABLE dbo.OrderDetail
(
  OrderDetailID INT IDENTITY(1, 1) NOT NULL
  ,OrderHeaderID INT NOT NULL
  ,ProductID INT NOT NULL
  ,RigNumber AS (OrderDetailID * 2)
  ,UnitPrice MONEY DEFAULT 0 NOT NULL
  PRIMARY KEY(OrderDetailID)
);
GO

IF OBJECT_ID('dbo.OrderHeader', 'U') IS NOT NULL
  DROP TABLE dbo.OrderHeader;
GO

CREATE TABLE dbo.OrderHeader
(
  OrderID INT IDENTITY(1, 1) NOT NULL
  ,OrderDATE DATETIME DEFAULT GETDATE() NOT NULL
  ,OrderNUMBER AS (ISNULL(N'SO' + CONVERT([nvarchar](23), [OrderID], 0), N'*** ERROR ***'))
  ,CustomerID INT DEFAULT 1 NOT NULL
  ,ShipName VARCHAR(20) DEFAULT 'Name'
  ,ShipAddress VARCHAR(40) DEFAULT 'Address'
  ,ShipVia VARCHAR(40) DEFAULT 'Via'
  ,ShipCity VARCHAR(20) DEFAULT 'City'
  ,ShipRegion VARCHAR(20) DEFAULT 'Region'
  ,ShipPostalCode VARCHAR(20) DEFAULT 'Postal code'
  ,ShipCountry VARCHAR(20) DEFAULT 'Country'
  PRIMARY KEY(OrderID)
);
GO

ALTER TABLE dbo.OrderDetail WITH CHECK
  ADD CONSTRAINT FK_OrderHeaderID FOREIGN KEY (OrderHeaderID) REFERENCES dbo.OrderHeader (OrderID);
GO

CREATE NONCLUSTERED INDEX IDX_OrderHeader_CustomerID ON dbo.OrderHeader
(
  CustomerID ASC
);
GO

DECLARE @i AS INTEGER = 0;
--219131
WHILE (@i < 250000)
BEGIN
  SET NOCOUNT ON;

  -- Insert dbo.OrderHeader
  INSERT INTO dbo.OrderHeader DEFAULT VALUES;

  -- Insert OrderDetail
  INSERT INTO dbo.OrderDetail
  (OrderHeaderID, ProductID, UnitPrice)
  SELECT (SELECT MAX(OrderID) FROM dbo.OrderHeader), (@i * 2)+1, ((@i * 2)+1)/2;

  SET @i = (@i + 1);

  SET NOCOUNT ON;
END;
GO

-- Delete the detail for the order ID number 1100
DELETE FROM dbo.OrderDetail WHERE OrderHeaderID=1100;
GO

/*
SELECT * FROM dbo.OrderDetail WHERE OrderHeaderID=1100;
SELECT * FROM dbo.OrderHeader WHERE NOT EXISTS (SELECT OrderHeaderID FROM dbo.OrderDetail WHERE OrderDetail.OrderHeaderID = OrderHeader.OrderID)
*/

------------------------------------------------------------------------
-- User defined functions                                              -
------------------------------------------------------------------------

/*
IF OBJECT_ID('Production.udf_Get_BillOfMaterials_Tree') IS NOT NULL
  DROP FUNCTION Production.udf_Get_BillOfMaterials_Tree;
GO
*/

CREATE OR ALTER FUNCTION Production.udf_Get_BillOfMaterials_Tree
(
  @Input_Product INTEGER
)
RETURNS TABLE
AS
RETURN
(
-- BillOfMaterials NON OTTIMIZZATA
-- Level 0
SELECT
  NULL CodPadre
  ,NULL IDPadre
  ,P0.Name CodFiglio
  ,P0.ProductID IDFiglio
  ,CAST(CAST(LTRIM(RTRIM(STR(P0.ProductID))) AS VARCHAR(20)) AS VARCHAR(MAX)) as Path
  ,B0.BillOfMaterialsID
  ,0 AS Livello
FROM
  Production.BillOfMaterials B0
JOIN
  Production.Product P0 ON P0.ProductID=B0.ComponentID
WHERE
  -- Parent product identification number
  -- Foreign key to Product.ProductID
  (B0.ProductAssemblyID IS NULL)
  -- Input filter
  AND (B0.ComponentID = @Input_Product)
  AND GETDATE() BETWEEN B0.StartDate AND ISNULL(B0.EndDate, '99991231')

UNION

-- Level 1
SELECT
  P0.Name CodPadre
  ,P0.ProductID IDPadre
  ,P1.Name CodFiglio
  ,P1.ProductID IDFiglio
  ,CAST(CAST(LTRIM(RTRIM(STR(P0.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P1.ProductID))) AS VARCHAR(20)) AS VARCHAR(MAX)) as Path
  ,B1.BillOfMaterialsID
  ,1 AS Livello
FROM
  Production.BillOfMaterials B0
JOIN
  Production.Product P0 ON P0.ProductID=B0.ComponentID
JOIN
  Production.BillOfMaterials B1 ON B1.ProductAssemblyID=B0.ComponentID
JOIN
  Production.Product P1 ON P1.ProductID=B1.ComponentID
WHERE
  (B0.ProductAssemblyID IS NULL)
  -- Input filter
  AND (B0.ComponentID = @Input_Product)
  AND GETDATE() BETWEEN B1.StartDate AND ISNULL(B1.EndDate, '99991231')

UNION

-- Level 2
SELECT
  P1.Name CodPadre
  ,P1.ProductID IDPadre
  ,P2.Name CodFiglio
  ,P2.ProductID IDFiglio
  ,CAST(CAST(LTRIM(RTRIM(STR(P0.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P1.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P2.ProductID))) AS VARCHAR(20)) AS VARCHAR(MAX)) as Path
  ,B2.BillOfMaterialsID
  ,2 AS Livello
FROM
  Production.BillOfMaterials B0
JOIN
  Production.Product P0 ON P0.ProductID=B0.ComponentID
JOIN
  Production.BillOfMaterials B1 ON B1.ProductAssemblyID=B0.ComponentID
JOIN
  Production.Product P1 ON P1.ProductID=B1.ComponentID
JOIN
  Production.BillOfMaterials B2 ON B2.ProductAssemblyID=B1.ComponentID
JOIN
  Production.Product P2 ON P2.ProductID=B2.ComponentID
WHERE
  -- Parent product identification number
  -- Foreign key to Product.ProductID
  (B0.ProductAssemblyID IS NULL)
  -- Input filter
  AND (B0.ComponentID = @Input_Product)
  AND GETDATE() BETWEEN B2.StartDate AND ISNULL(B2.EndDate, '99991231')

UNION

-- Level 3
SELECT
  P2.Name CodPadre
  ,P2.ProductID IDPadre
  ,P3.Name CodFiglio
  ,P3.ProductID IDFiglio
  ,CAST(CAST(LTRIM(RTRIM(STR(P0.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P1.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P2.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P3.ProductID))) AS VARCHAR(20)) AS VARCHAR(MAX)) as Path
  ,B3.BillOfMaterialsID
  ,3 AS Livello
FROM
  Production.BillOfMaterials B0
JOIN
  Production.Product P0 ON P0.ProductID=B0.ComponentID
JOIN
  Production.BillOfMaterials B1 ON B1.ProductAssemblyID=B0.ComponentID
JOIN
  Production.Product P1 ON P1.ProductID=B1.ComponentID
JOIN
  Production.BillOfMaterials B2 ON B2.ProductAssemblyID=B1.ComponentID
JOIN
  Production.Product P2 ON P2.ProductID=B2.ComponentID
JOIN
  Production.BillOfMaterials B3 ON B3.ProductAssemblyID=B2.ComponentID
JOIN
  Production.Product P3 ON P3.ProductID=B3.ComponentID
WHERE
  (B0.ProductAssemblyID IS NULL)
  -- Input filter
  AND (B0.ComponentID = @Input_Product)
  AND GETDATE() BETWEEN B3.StartDate AND ISNULL(B3.EndDate, '99991231')

UNION

-- Level 4
SELECT
  P3.Name CodPadre
  ,P3.ProductID IDPadre
  ,P4.Name CodFiglio
  ,P4.ProductID IDFiglio
  ,CAST(CAST(LTRIM(RTRIM(STR(P0.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P1.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P2.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P3.ProductID))) AS VARCHAR(20)) + '.' + CAST(LTRIM(RTRIM(STR(P4.ProductID))) AS VARCHAR(20)) AS VARCHAR(MAX)) as Path
  ,B4.BillOfMaterialsID
  ,4 AS Livello
FROM
  Production.BillOfMaterials B0
JOIN
  Production.Product P0 ON P0.ProductID=B0.ComponentID
JOIN
  Production.BillOfMaterials B1 ON B1.ProductAssemblyID=B0.ComponentID
JOIN
  Production.Product P1 ON P1.ProductID=B1.ComponentID
JOIN
  Production.BillOfMaterials B2 ON B2.ProductAssemblyID=B1.ComponentID
JOIN
  Production.Product P2 ON P2.ProductID=B2.ComponentID
JOIN
  Production.BillOfMaterials B3 ON B3.ProductAssemblyID=B2.ComponentID
JOIN
  Production.Product P3 ON P3.ProductID=B3.ComponentID
JOIN
  Production.BillOfMaterials B4 ON B4.ProductAssemblyID=B3.ComponentID
JOIN
  Production.Product P4 ON P4.ProductID=B4.ComponentID
WHERE
  (B0.ProductAssemblyID IS NULL)
  -- Input filter
  AND (B0.ComponentID = @Input_Product)
  AND GETDATE() BETWEEN B4.StartDate AND ISNULL(B4.EndDate, '99991231')
);
GO